#include <hidef.h> /* for EnableInterrupts macro */
#include <stdio.h>
#include "main.h"
#include "derivative.h" /* include peripheral declarations */
#include "hardware.h"
#include "i2c.h"

static void waitBit(void)
{
		asm {
			nop
			nop
			nop
			nop
			nop
			nop
			nop
			nop
			nop
			nop
			nop
			nop
		}
}

static void setSda(uint8_t state)
{
	if (state) PTBD_PTBD0 = 1; else PTBD_PTBD0 = 0;
  
}

static void setScl(uint8_t state)
{
	if (state) PTBD_PTBD1 = 1; else PTBD_PTBD1 = 0;
}

static uint8_t isSda()
{
  uint8_t result;
  PTBDD &= ~1;
	result = PTBD_PTBD0;
  PTBDD |= 1;
  return result;
}

static void i2cStart(void)
{
	waitBit();
	setSda(1);
	setScl(1);
	waitBit();
	setSda(0);
	waitBit();
	setScl(0);
	waitBit();
}

static void i2cSendByte(unsigned char b)
{
	int i;
	for (i = 0; i < 8; i++) {
		waitBit();
		setScl(0);
		setSda((b & 128) ? 1 : 0);
		waitBit();
		setScl(1);
		waitBit();
		waitBit();
		setScl(0);
		b <<= 1;
	}
	setSda(1);
	waitBit();
}

static int testAcknowledge(void)
{
	int result;
	setScl(0);
	setSda(1);
	waitBit();
	setScl(1);
	waitBit();
	result = isSda();
	waitBit();
	setScl(0);
	waitBit();
	return !result;
}

static void sendAcknowledge(void)
{
	setScl(0);
	setSda(0);
	waitBit();
	setScl(1);
	waitBit();
	setScl(0);
	setSda(1);
	waitBit();
}

static void sendNotAcknowledge(void)
{
	setScl(0);
	setSda(1);
	waitBit();
	setScl(1);
	waitBit();
	setScl(0);
	setSda(1);
	waitBit();
}

static void i2cStop(void)
{
	waitBit();
	setSda(0);
	setScl(0);
	waitBit();
	setScl(1);
	waitBit();
	setSda(1);
	waitBit();
}

static uint8_t i2cReadByte()
{
  uint8_t result = 0;
	int i;
	setSda(1);
	for (i = 0; i < 8; i++) {
		waitBit();
		setScl(0);
		waitBit();
		setScl(1);
		result <<= 1;
		if (isSda()) result |= 1;
		waitBit();
		setScl(0);
	}
	waitBit();
	return result;
}

static int calculateLm75Temperature(int data)
{
	const int bits = 13;
	const int msb = 1 << (bits - 1);
	const int max = 2*msb;
	if (data >= max || data < 0) return 0;
	if (data & msb) data = data - max;
	return data * 10 / 32;
}

int16_t i2cReadTemperature()
{
  uint16_t data;
  i2cStart();
  i2cSendByte(145);  // I2C address 72, read-mode
  testAcknowledge();
  data = i2cReadByte() << 5;
	sendAcknowledge();
  data |= i2cReadByte() >> 3;
	sendNotAcknowledge();
  i2cStop();
  return calculateLm75Temperature(data);
}

static int16_t readAccelReg(uint8_t reg)
{
  int16_t result;
  i2cStart();
  i2cSendByte(28*2);  // I2C address 28, write-mode
  testAcknowledge();
  i2cSendByte(reg);
  testAcknowledge();
  i2cStart();
  i2cSendByte(28*2+1);  // I2C address 28, read-mode
  testAcknowledge();
  result = i2cReadByte() << 8;
	sendAcknowledge();
  result |= i2cReadByte();
	sendNotAcknowledge();
  i2cStop();
  return result;
}

static void writeAccelReg(uint8_t reg, uint8_t value)
{
  i2cStart();
  i2cSendByte(28*2);  // I2C address 28, write-mode
  testAcknowledge();
  i2cSendByte(reg);
  testAcknowledge();
  i2cSendByte(value);
  testAcknowledge();
  i2cStop();
}

static uint8_t readGpioReg(uint8_t reg)
{
  int8_t result;
  i2cStart();
  i2cSendByte(32*2);  // I2C address 32, write-mode
  testAcknowledge();
  i2cSendByte(reg);
  testAcknowledge();
  i2cStart();
  i2cSendByte(32*2+1);  // I2C address 32, read-mode
  testAcknowledge();
  result = i2cReadByte();
	sendNotAcknowledge();
  i2cStop();
  return result;
}

static void writeGpioReg(uint8_t reg, uint8_t value)
{
  i2cStart();
  i2cSendByte(32*2);  // I2C address 32, write-mode
  testAcknowledge();
  i2cSendByte(reg);
  testAcknowledge();
  i2cSendByte(value);
  testAcknowledge();
  i2cStop();
}

void i2cTest(void)
{
	for (;;) {
    i2cGpioSet(0);
	  PTBD_PTBD4 = 1;

  	// LM75 temperature
	  printf("%i,", i2cReadTemperature());
	  
	  // accelerometer
	  printf("%i,%i,%i\r\n", i2cReadAccelX(), i2cReadAccelY(), i2cReadAccelZ());
	  
//void i2cGpioConfigureInputPins(uint8_t port0, uint8_t port1);

	  printf("gpio: %04x\r\n", i2cGpioGet());
	  
	  
  	delayMs(400);
    i2cGpioSet(0x8000);
	  PTBD_PTBD4 = 0;

  	delayMs(500);
	}
}

void i2cInit()
{
  int i;

	// init pins
	setSda(1);
  setScl(1);
    
	// activate accelerometer
  writeAccelReg(0x2a, 1);

  // configure P1.7 as output
  i2cGpioConfigureInputPins(~(1 << 15));

	// test addresses
 	printf("testing available I2C devices...\r\n");
 	for (i = 1; i < 256; i++) {
 		int ack;
 		i2cStart();
 		i2cSendByte((uint8_t) i);
 		ack = testAcknowledge();
	  if (i & 1) {
	    i2cReadByte();
  	  sendNotAcknowledge();
	  }
 		i2cStop();
 		if (ack) {
 			printf("I2C %s acknowledge at address: %i\r\n", (i & 1) ? "read" : "write", i / 2);
 		}
 	}
}

int16_t i2cReadAccelX()
{
  return readAccelReg(1);  
}

int16_t i2cReadAccelY()
{
  return readAccelReg(3);  
}

int16_t i2cReadAccelZ()
{
  return readAccelReg(5);  
}

void i2cGpioConfigureInputPins(uint16_t inputMask)
{
  writeGpioReg(6, inputMask & 0xff);
  writeGpioReg(7, inputMask >> 8);
}

uint16_t i2cGpioGet()
{
  return readGpioReg(0) | (readGpioReg(1) << 8);
}

void i2cGpioSet(uint16_t bits)
{
  writeGpioReg(2, bits & 0xff);
  writeGpioReg(3, bits >> 8);
}
