#include <stdio.h>
#include "main.h"
#include "hardware.h"
#include "rfmodule.h"

// RF module defines
#define CTRL_X (0x18)
#define CTRL_S (0x13)
#define RX_FIFO_START (0x300)
#define RX_FIFO_DATA_START (0x30C)

//Short Address Control Registers for writing
#define WRITE_RXMCR (0x01)
#define WRITE_PANIDL (0x03)
#define WRITE_PANIDH (0x05)
#define WRITE_SADRL (0x07)
#define WRITE_SADRH (0x09)
#define WRITE_EADR0 (0x0B)
#define WRITE_RXFLUSH (0x1B)
#define WRITE_PACON2 (0x31)
#define WRITE_TXNCON (0x37)
#define WRITE_SOFTRST (0x55)
#define WRITE_TXSTBL (0x5D)
#define WRITE_INTCON (0x65)
#define WRITE_RFCTL (0x6D)
#define WRITE_BBREG2 (0x75)
#define WRITE_BBREG6 (0x7D)
#define WRITE_CCAEDTH (0x7F)

#define RSSITHCCA (0x3F)

//Short Address Control Registers for reading
#define READ_TXSTAT (0x48)
#define READ_SOFTRST (0x54)
#define READ_INTSTAT (0x62)

//Long Address Control Registers
#define RFCON0 (0x200)
#define RFCON1 (0x201)
#define RFCON2 (0x202)
#define RFCON3 (0x203)
#define RFCON6 (0x206)
#define RFCON7 (0x207)
#define RFCON8 (0x208)
#define RFSTATE (0x20f)
#define SLPCON1 (0x220)

#define CHANNEL_11 (0x00)
#define CHANNEL_12 (0x10)
#define CHANNEL_13 (0x20)
#define CHANNEL_14 (0x30)
#define CHANNEL_15 (0x40)
#define CHANNEL_16 (0x50)
#define CHANNEL_17 (0x60)
#define CHANNEL_18 (0x70)
#define CHANNEL_19 (0x80)
#define CHANNEL_20 (0x90)
#define CHANNEL_21 (0xa0)
#define CHANNEL_22 (0xb0)
#define CHANNEL_23 (0xc0)
#define CHANNEL_24 (0xd0)
#define CHANNEL_25 (0xe0)
#define CHANNEL_26 (0xf0)

static void setShortRAMAddr(uint8_t address, uint8_t value)
{
	//Enable chip select
	setCS(0);
	spiTransfer(address);
	/* write data */
	spiTransfer(value);
	//Disable chip select
	setCS(1);
}

static void setLongRAMAddr(uint16_t address, uint8_t value)
{
  uint8_t temp;
	//Enable chip select
	setCS(0);
	temp = (((uint8_t) (address >> 3)) & 0x7F) | 0x80;
	spiTransfer(temp);
	temp = (((uint8_t) (address << 5)) & 0xE0) | 0x10;
	spiTransfer(temp);
	spiTransfer(value);
	//Disable chip select
	setCS(1);
}

static uint8_t getShortRAMAddr(uint8_t address)
{
	uint8_t result;
	//Enable chip select
	setCS(0);
	spiTransfer(address);
	result = spiTransfer(0);
	//Disable chip select
	setCS(1);
	return result;
}

static uint8_t getLongRAMAddr(uint16_t address)
{
	uint8_t result, temp;
	//Enable chip select
	setCS(0);
	temp = ((address >> 3) & 0x7F) | 0x80;
	spiTransfer(temp);
	temp = ((address << 5) & 0xE0);
	spiTransfer(temp);
	result = spiTransfer(0);
	//Disable chip select
	setCS(1);
	return result;
}

void rfInit()
{
	const unsigned char longAddress[8] = { 0x07, 0x06, 0x05, 0x04, 0x03, 0x02,
		0x01, 0x00 };
	setCS(1);
	setReset(0);
	delayMs(300);
	setReset(1);
	delayMs(300);

	// Reset the RF module
	setShortRAMAddr(WRITE_RFCTL, 0x01);
	// Remove the RF module from reset
	setShortRAMAddr(WRITE_RFCTL, 0x00);

	// Flush the RX fifo
	setShortRAMAddr(WRITE_RXFLUSH, 0x01);

	// Enable the RF-PLL
	setLongRAMAddr(RFCON2, 0x80);
	// Set TX for max output power
	setLongRAMAddr(RFCON3, 0x00);
	// Enabled TX filter control
	setLongRAMAddr(RFCON6, 0x80);
	setLongRAMAddr(RFCON8, 0b00010000);
	// Program CCA mode using RSSI
	setShortRAMAddr(WRITE_BBREG2, 0x78);

	// Enable the packet RSSI
	setShortRAMAddr(WRITE_BBREG6, 0x40);
	// Program CCA, RSSI threshold values
	setShortRAMAddr(RSSITHCCA, 0x00);

	//Enable only TXNIF and RXIF interrupts
	setShortRAMAddr(WRITE_INTCON, 0xF6);
	
	// disable automatic acknowledge and enable promiscuous mode
	setShortRAMAddr(WRITE_RXMCR, (1 << 5) | 1);

	// Set channel
	setLongRAMAddr(RFCON0, CHANNEL_11);
	//Reset the RF module with new settings
	setShortRAMAddr(WRITE_RFCTL, 0x04);
	setShortRAMAddr(WRITE_RFCTL, 0x00);
}

uint8_t rfGetPacketLength(void)
{
	//read the interrupt status register to see what caused the interrupt
	uint8_t flags = getShortRAMAddr(READ_INTSTAT);
	uint8_t length;

	if (flags && (1 << 3)) {
		setShortRAMAddr(WRITE_RXFLUSH, 0x01);
  	length = getLongRAMAddr(RX_FIFO_START) - 6;
  	return length;
	}
	return 0;
}

void rfGetData(void* data, uint8_t size)
{
  uint8_t i;
  for (i = 0; i < size; i++) ((uint8_t*)data)[i] = getLongRAMAddr(RX_FIFO_START + 4 + i);
}

void rfSendData(void* data, uint8_t size)
{
  static uint8_t sequenceID = 0;
  uint8_t i;
  setLongRAMAddr(0x000, 0x00); //Security header length
  setLongRAMAddr(0x001, (size + 0x04)); //Length
  setLongRAMAddr(0x002, 0x01); //Frame control, Data command, no acknowledge
  setLongRAMAddr(0x003, 0x00); //Frame control, no address
  setLongRAMAddr(0x004, sequenceID); //Sequence id
  for (i = 0; i < size; i++) setLongRAMAddr(i + 5, ((uint8_t*)data)[i]);

	setShortRAMAddr(WRITE_RXFLUSH, 0x01);
  setShortRAMAddr(WRITE_TXNCON, 0x01); //transmit packet without ACK requested
  
  // wait until packet is sent
  for (;;) {
  	uint8_t flags = getShortRAMAddr(READ_INTSTAT);
    if (flags & 1) break;
  }
	setShortRAMAddr(WRITE_RXFLUSH, 0x01);

  sequenceID++;
}
